/*
 * Copyright 2017 Lutz Fischer <lfischer@staffmail.ed.ac.uk>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package rappsilber.ms.sequence;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.stream.Collectors;
import rappsilber.utils.MyArrayUtils;

/**
 * A representation of a modified peptide
 * @author Lutz Fischer <lfischer@staffmail.ed.ac.uk>
 */
public class ModifiedPeptide extends Peptide {
    /**
     * the non-modified version of this peptide
     */
    Peptide basePeptide;
    /**
     * list of modifications applied to this peptide
     */
    HashMap<Integer,Modification[]> modifications= new HashMap<>();
    /**
     * Mass difference between this peptide and the unmodified version of this 
     * peptide
     */
    double deltaMass;

    /**
     * Constructor. Creates a modified peptide based on the base-peptide.
     * If the base peptide is an instance of ModifiedPeptide the modifications 
     * are taken over
     * Otherwise no modifications are defined
     * @param base 
     */
    public ModifiedPeptide(Peptide base) {
        super(base instanceof ModifiedPeptide? ((ModifiedPeptide) base).basePeptide : base);
        if (base instanceof ModifiedPeptide) {
            ModifiedPeptide mp = (ModifiedPeptide) base;
            this.basePeptide=mp;
            modifications = (HashMap<Integer, Modification[]>) mp.modifications.clone();
        } else {
            base = basePeptide;
        }
        deltaMass=0;
    }

    /**
     * Constructor. Creates a modified peptide based on the base-peptide.
     * the new modifications are replacing any old ones
     * @param base 
     */
    public ModifiedPeptide(Peptide base,HashMap<Integer,Modification[]> mods) {
        this(base);
        deltaMass=0;
        this.modifications = (HashMap<Integer, Modification[]>) mods.clone();
        for (Modification[] ma : mods.values()) {
            for (Modification m : ma) {
                deltaMass+=m.getMass();
            }
        }
    }

    /**
     * Constructor. Creates a modified peptide based on the base-peptide.
     * In case when the base peptide is a modified peptide then the new 
     * modification will be added to existing modifications
     * @param base 
     */
    public ModifiedPeptide(Peptide base,int pos, Modification mod) {
        this(base);
        Modification[] preior = modifications.get(pos);
        if (preior == null) {
            modifications.put(pos,new Modification[]{mod});
        } else {
            Modification[] newMods  = new Modification[preior.length+1];
            System.arraycopy(preior, 0, newMods, 0, preior.length);
            newMods[preior.length] = mod;
        }
        deltaMass+=mod.getMass();
    }
    
    /**
     * creates a new modified peptide having all modifications of the current 
     * peptide + the newly defined one
     * @param pos
     * @param mod
     * @return 
     */
    public ModifiedPeptide modify(int pos, Modification mod) {
        return new ModifiedPeptide(this,pos,mod);
    }

//    @Override
//    public String toString() {
//        StringBuilder sb = new StringBuilder(basePeptide.getLength()+modifications.size()*2);
//        Modification[] nterm = modifications.get(-1);
//        if (nterm != null) {
//            sb.append(Arrays.stream(nterm)
//            .map(s -> s.getSymbol())
//            .collect(Collectors.joining(".")));
//            sb.append(".");
//        }
//        for (int i = 0; i<length(); i++) {
//            sb.append(aminoAcidAt(i).SequenceID);
//            Modification[] mod = modifications.get(-1);
//            if (mod != null) {
//                sb.append(Arrays.stream(mod)
//                .map(s -> s.getSymbol())
//                .collect(Collectors.joining(".")));
//            }
//        }
//        Modification[] cterm = modifications.get(length());
//        if (cterm != null) {
//            sb.append(".");
//            sb.append(Arrays.stream(cterm)
//            .map(s -> s.getSymbol())
//            .collect(Collectors.joining(".")));
//        }
//        
//        return super.toString(); //To change body of generated methods, choose Tools | Templates.
//    }

    @Override
    public double getMass() {
        return basePeptide.getMass()+deltaMass;
    }

    
    
    @Override
    public String toStringBaseSequence() {
        return basePeptide.toStringBaseSequence(); //To change body of generated methods, choose Tools | Templates.
    }
    
    
    
    
}
