/* 
 * Copyright 2016 Lutz Fischer <l.fischer@ed.ac.uk>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package rappsilber.gui.localapplication;

import java.awt.event.ActionEvent;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.ParseException;
import java.util.HashMap;
import java.util.logging.Filter;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JOptionPane;
import rappsilber.config.AbstractRunConfig;
import rappsilber.data.csv.CSVRandomAccess;
import rappsilber.gui.components.GenericTextPopUpMenu;
import rappsilber.gui.logging.JTextAreaHandle;
import rappsilber.gui.logging.JTextFieldHandle;
import rappsilber.ms.ToleranceUnit;
import rappsilber.ms.dataAccess.msm.MSMListIterator;
import rappsilber.ms.spectra.Spectra;
import rappsilber.utils.ColumnAlternatives;
import rappsilber.utils.Util;

/**
 *
 * @author Lutz Fischer <l.fischer@ed.ac.uk>
 */
public class RetentionTime extends javax.swing.JFrame {

    JTextAreaHandle m_loggingHandle;
    JTextFieldHandle m_statusHandle;
    Logger m_rappsilberLoger;
    public class SpectrumMetaData {
        double rtStart;
        double rtEnd;
        double precIntensity;
        double precMZ;
        double precCharge;
        double precMass;

        public SpectrumMetaData(Spectra s) {
            rtStart = s.getElutionTimeStart();
            rtEnd = s.getElutionTimeEnd();
            precIntensity = s.getPrecurserIntensity();
            precMZ = s.getPrecurserMZ();
            precCharge = s.getPrecurserCharge();
            precMass = s.getPrecurserMass();
        }
        
    }
    
    /**
     * Creates new form RetentionTime
     */
    public RetentionTime() {
        initComponents();
        m_loggingHandle = new JTextAreaHandle(jTextArea1);
        m_loggingHandle.setFilter(new Filter() {

            public boolean isLoggable(LogRecord record) {
                return true;
            }
        });

        m_statusHandle = new JTextFieldHandle(txtStatus);
        m_statusHandle.setFilter(new Filter() {

            public boolean isLoggable(LogRecord record) {
                return true;
            }
        });
        

        m_loggingHandle.setLevel(Level.ALL);
        m_statusHandle.setLevel(Level.ALL);

        m_rappsilberLoger = Logger.getLogger("rappsilber");
        //Logger.getLogger("").addHandler(m_loggingHandle);
        m_rappsilberLoger.addHandler(m_statusHandle);
        m_rappsilberLoger.addHandler(m_loggingHandle);
        m_rappsilberLoger.setLevel(Level.ALL);
        
        csvScans.setAutoLoadFile(true);
        
        csvScans.addHeaderChangedListener(new java.awt.event.ActionListener() {

            public void actionPerformed(ActionEvent e) {
                setHeaderComboBoxes();
            }
        });
        
        csvScans.setEditable(true);
        
        GenericTextPopUpMenu copyPaste = new GenericTextPopUpMenu();
        copyPaste.installContextMenu(this);
        
        
        
    }

    public void setHeaderComboBoxes() {
        CSVRandomAccess csv = csvScans.getCSV();
        ColumnAlternatives.setupAlternatives(csv);
        int cols = csv.getMaxColumns();
        String[] comboBoxHeaders = new String[cols+1];
        comboBoxHeaders[0] = "--Select--";
        for (int i = 1;i<comboBoxHeaders.length;i++) {
            comboBoxHeaders[i] = csv.getHeader(i - 1);
        }
        
        cbRun.setModel(new DefaultComboBoxModel(comboBoxHeaders));
        cbScan.setModel(new DefaultComboBoxModel(comboBoxHeaders));
        Integer runCol = csv.getColumn("run");
        if ( runCol != null)
            cbRun.setSelectedIndex(runCol+1);

        Integer scanCol = csv.getColumn("scan");
        if ( scanCol != null)
            cbScan.setSelectedIndex(scanCol+1);        
    }
    
    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jTabbedPane1 = new javax.swing.JTabbedPane();
        jPanel3 = new javax.swing.JPanel();
        csvScans = new rappsilber.gui.components.CSVPanel();
        jPanel4 = new javax.swing.JPanel();
        jPanel6 = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        cbRun = new javax.swing.JComboBox();
        jLabel2 = new javax.swing.JLabel();
        cbScan = new javax.swing.JComboBox();
        jPanel5 = new javax.swing.JPanel();
        jButton1 = new javax.swing.JButton();
        jPanel2 = new javax.swing.JPanel();
        flmgfFiles = new rappsilber.gui.components.FileList();
        jPanel1 = new javax.swing.JPanel();
        jScrollPane1 = new javax.swing.JScrollPane();
        jTextArea1 = new javax.swing.JTextArea();
        txtStatus = new javax.swing.JTextField();

        setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);

        jPanel3.setLayout(new java.awt.BorderLayout());

        csvScans.setHasHeader(true);
        csvScans.setMaximumSize(new java.awt.Dimension(2147483647, 30));
        csvScans.setMinimumSize(new java.awt.Dimension(252, 30));
        csvScans.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                csvScansActionPerformed(evt);
            }
        });
        jPanel3.add(csvScans, java.awt.BorderLayout.CENTER);

        jPanel4.setMaximumSize(new java.awt.Dimension(32767, 30));
        jPanel4.setLayout(new java.awt.BorderLayout());

        jLabel1.setText("Run Column");

        cbRun.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "--Select--" }));

        jLabel2.setText("Scan Column");

        cbScan.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "--Select--" }));

        javax.swing.GroupLayout jPanel6Layout = new javax.swing.GroupLayout(jPanel6);
        jPanel6.setLayout(jPanel6Layout);
        jPanel6Layout.setHorizontalGroup(
            jPanel6Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, jPanel6Layout.createSequentialGroup()
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addComponent(jLabel2)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(cbScan, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jLabel1)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(cbRun, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        );
        jPanel6Layout.setVerticalGroup(
            jPanel6Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel6Layout.createSequentialGroup()
                .addGap(35, 35, 35)
                .addGroup(jPanel6Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel1)
                    .addComponent(cbRun, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel2)
                    .addComponent(cbScan, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(45, Short.MAX_VALUE))
        );

        jPanel4.add(jPanel6, java.awt.BorderLayout.WEST);

        jPanel5.setMaximumSize(new java.awt.Dimension(32767, 30));
        jPanel5.setPreferredSize(new java.awt.Dimension(605, 30));

        jButton1.setText("Run");
        jButton1.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButton1ActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout jPanel5Layout = new javax.swing.GroupLayout(jPanel5);
        jPanel5.setLayout(jPanel5Layout);
        jPanel5Layout.setHorizontalGroup(
            jPanel5Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, jPanel5Layout.createSequentialGroup()
                .addGap(0, 544, Short.MAX_VALUE)
                .addComponent(jButton1))
        );
        jPanel5Layout.setVerticalGroup(
            jPanel5Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, jPanel5Layout.createSequentialGroup()
                .addContainerGap(47, Short.MAX_VALUE)
                .addComponent(jButton1)
                .addGap(32, 32, 32))
        );

        jPanel4.add(jPanel5, java.awt.BorderLayout.EAST);

        jPanel3.add(jPanel4, java.awt.BorderLayout.SOUTH);

        jTabbedPane1.addTab("Scans", jPanel3);

        flmgfFiles.setDescription("mgf-files");
        flmgfFiles.setExtensions(new String[] {"mgf"});

        javax.swing.GroupLayout jPanel2Layout = new javax.swing.GroupLayout(jPanel2);
        jPanel2.setLayout(jPanel2Layout);
        jPanel2Layout.setHorizontalGroup(
            jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel2Layout.createSequentialGroup()
                .addComponent(flmgfFiles, javax.swing.GroupLayout.DEFAULT_SIZE, 617, Short.MAX_VALUE)
                .addContainerGap())
        );
        jPanel2Layout.setVerticalGroup(
            jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel2Layout.createSequentialGroup()
                .addComponent(flmgfFiles, javax.swing.GroupLayout.DEFAULT_SIZE, 345, Short.MAX_VALUE)
                .addContainerGap())
        );

        jTabbedPane1.addTab("MGF-Files", jPanel2);

        jTextArea1.setColumns(20);
        jTextArea1.setRows(5);
        jScrollPane1.setViewportView(jTextArea1);

        javax.swing.GroupLayout jPanel1Layout = new javax.swing.GroupLayout(jPanel1);
        jPanel1.setLayout(jPanel1Layout);
        jPanel1Layout.setHorizontalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jScrollPane1, javax.swing.GroupLayout.DEFAULT_SIZE, 605, Short.MAX_VALUE)
                .addContainerGap())
        );
        jPanel1Layout.setVerticalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jScrollPane1, javax.swing.GroupLayout.DEFAULT_SIZE, 333, Short.MAX_VALUE)
                .addContainerGap())
        );

        jTabbedPane1.addTab("Log", jPanel1);

        txtStatus.setEditable(false);

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jTabbedPane1, javax.swing.GroupLayout.DEFAULT_SIZE, 634, Short.MAX_VALUE)
                .addContainerGap())
            .addComponent(txtStatus)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jTabbedPane1, javax.swing.GroupLayout.DEFAULT_SIZE, 384, Short.MAX_VALUE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(txtStatus, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void csvScansActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_csvScansActionPerformed

        setHeaderComboBoxes();
    }//GEN-LAST:event_csvScansActionPerformed

    private void jButton1ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButton1ActionPerformed

        Runnable runnable = new Runnable() {
            public void run() {
                jButton1.setEnabled(false);
                try {
                    jButton1.setEnabled(false);
                    HashMap<String, HashMap<Integer, SpectrumMetaData>> metaData = new HashMap<String, HashMap<Integer, SpectrumMetaData>>();
                    HashMap<String, HashMap<Integer, Double>> retentionStart = new HashMap<String, HashMap<Integer, Double>>();
                    HashMap<String, HashMap<Integer, Double>> retentionEnd = new HashMap<String, HashMap<Integer, Double>>();
                    readInRetentionTimes(metaData);
                    CSVRandomAccess csv = csvScans.getCSV();
                    int runCol = cbRun.getSelectedIndex()-1;
                    int scanCol = cbScan.getSelectedIndex()-1;
                    int retStart = csv.addGetColumn("ElutionStart");
                    int retEnd = csv.addGetColumn("ElutionEnd");
                    int spmz = csv.addGetColumn("SpectrumPrecursorMZ");
                    int spz = csv.addGetColumn("SpectrumPrecursorCharge");
                    int spm = csv.addGetColumn("SpectrumPrecursorMass");
                    int spi = csv.addGetColumn("SpectrumPrecursorIntensity");
                    csvScans.csvChanged();
                    //go through the csv
                    for (int i = 0; i < csv.getRowCount(); i++) {
                        String run = csv.getValue(runCol, i).toLowerCase();
                        String scan = csv.getValue(scanCol, i);
                        int sn = Integer.MIN_VALUE;
                        try {
                            sn = Integer.parseInt(scan);
                        } catch (NumberFormatException nfe) {
                            Logger.getLogger(this.getClass().getName()).log(Level.WARNING,scan + " is not recognised as number (scan-number)");
                            continue;
                        }
                        HashMap<Integer, SpectrumMetaData> smds = metaData.get(run);
                        if (smds == null) {
                            if (run.toLowerCase().endsWith(".raw")) {
                                smds = metaData.get(run.substring(0, run.length() - 4));
                            } else {
                                smds = metaData.get(run + ".raw");
                            }
                        }
                        if (smds != null) {
                            SpectrumMetaData smd = smds.get(sn);
                            if (smd != null) {
                                csv.setValue(smd.rtStart, retStart, i);
                                csv.setValue(smd.rtEnd, retEnd, i);
                                csv.setValue(smd.precMZ, spmz, i);
                                csv.setValue(smd.precCharge, spz, i);
                                csv.setValue(smd.precMass, spm, i);
                                csv.setValue(smd.precIntensity, spi, i);
                            }
                        }
                    }
                    
                    Logger.getLogger(this.getClass().getName()).log(Level.SEVERE, "done");
                } catch (FileNotFoundException ex) {
                    Logger.getLogger(this.getClass().getName()).log(Level.WARNING, "File not found", ex);
                    JOptionPane.showMessageDialog(rootPane, ex.getLocalizedMessage(), "file not found", JOptionPane.ERROR_MESSAGE);
                } catch (IOException ex) {
                    Logger.getLogger(this.getClass().getName()).log(Level.SEVERE, "Error wile reading file " , ex);
                    JOptionPane.showMessageDialog(rootPane, ex.getLocalizedMessage(), "Error wile reading file ", JOptionPane.ERROR_MESSAGE);
                } catch (ParseException ex) {
                    Logger.getLogger(this.getClass().getName()).log(Level.SEVERE, null, ex);
                    JOptionPane.showMessageDialog(rootPane, ex.getLocalizedMessage(), "Error wile reading file ", JOptionPane.ERROR_MESSAGE);
                }                
                jButton1.setEnabled(true);

            }

            protected void readInRetentionTimes(HashMap<String, HashMap<Integer, SpectrumMetaData>>  smd) throws IOException, FileNotFoundException, ParseException {
                MSMListIterator iter;
                ToleranceUnit t = new ToleranceUnit("100ppm");
                iter = new MSMListIterator(t, 0, new AbstractRunConfig() {
                });
                File[] list = flmgfFiles.getFiles();
                if (list.length > 0) {
                    for (File f : list) {
                        iter.addFile(f.getAbsolutePath(), "", t);
                    }
                }
                
                iter.init();
                
                while (iter.hasNext()) {
                    Spectra s = iter.next();
                    String run = s.getRun().toLowerCase();
                    int scan = s.getScanNumber();
                    double rets = s.getElutionTimeStart();
                    double rete = s.getElutionTimeEnd();
                    HashMap<Integer, SpectrumMetaData> runSmd = smd.get(run);
                    if (runSmd == null) {
                        runSmd = new HashMap<Integer, SpectrumMetaData>();
                        smd.put(run, runSmd);
                    }
                    runSmd.put(scan, new SpectrumMetaData(s));
                }
            }
        };

        if (csvScans.getCSV() == null || csvScans.getCSV().getRowCount() == 0) {
            JOptionPane.showMessageDialog(rootPane, "No scans loaded");
            csvScans.requestFocusInWindow();
            return;
        }

        
        if (cbScan.getSelectedIndex() == 0) {
            JOptionPane.showMessageDialog(rootPane, "Select a column with the scan numbers!");
            cbScan.requestFocusInWindow();
            return;
        }
        
        if (cbRun.getSelectedIndex() == 0) {
            JOptionPane.showMessageDialog(rootPane, "Select a column with the run-names!");
            cbRun.requestFocusInWindow();
            return;
        }

        if (flmgfFiles.getFiles().length == 0) {
            JOptionPane.showMessageDialog(rootPane, "No mgf files selected");
            flmgfFiles.requestFocusInWindow();
            return;
        }
        
        
        new Thread(runnable).start();
    }//GEN-LAST:event_jButton1ActionPerformed

    /**
     * @param args the command line arguments
     */
    public static void main(String args[]) {
        /* Set the Nimbus look and feel */
        //<editor-fold defaultstate="collapsed" desc=" Look and feel setting code (optional) ">
        /* If Nimbus (introduced in Java SE 6) is not available, stay with the default look and feel.
         * For details see http://download.oracle.com/javase/tutorial/uiswing/lookandfeel/plaf.html 
         */
        try {
            for (javax.swing.UIManager.LookAndFeelInfo info : javax.swing.UIManager.getInstalledLookAndFeels()) {
                if ("Nimbus".equals(info.getName())) {
                    javax.swing.UIManager.setLookAndFeel(info.getClassName());
                    break;
                }
            }
        } catch (ClassNotFoundException ex) {
            java.util.logging.Logger.getLogger(RetentionTime.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        } catch (InstantiationException ex) {
            java.util.logging.Logger.getLogger(RetentionTime.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        } catch (IllegalAccessException ex) {
            java.util.logging.Logger.getLogger(RetentionTime.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        } catch (javax.swing.UnsupportedLookAndFeelException ex) {
            java.util.logging.Logger.getLogger(RetentionTime.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        }
        //</editor-fold>

        /* Create and display the form */
        java.awt.EventQueue.invokeLater(new Runnable() {
            public void run() {
                new RetentionTime().setVisible(true);
            }
        });
    }
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JComboBox cbRun;
    private javax.swing.JComboBox cbScan;
    private rappsilber.gui.components.CSVPanel csvScans;
    private rappsilber.gui.components.FileList flmgfFiles;
    private javax.swing.JButton jButton1;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JPanel jPanel3;
    private javax.swing.JPanel jPanel4;
    private javax.swing.JPanel jPanel5;
    private javax.swing.JPanel jPanel6;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JTabbedPane jTabbedPane1;
    private javax.swing.JTextArea jTextArea1;
    private javax.swing.JTextField txtStatus;
    // End of variables declaration//GEN-END:variables
}
