/* 
 * Copyright 2016 Lutz Fischer <l.fischer@ed.ac.uk>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package rappsilber.gui.components;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import javax.swing.JFrame;
import rappsilber.config.LocalProperties;
import rappsilber.gui.GetFile;

/**
 *
 * @author Lutz Fischer <l.fischer@ed.ac.uk>
 */
public class FileBrowser extends javax.swing.JPanel {
    private static final long serialVersionUID = 676003385063269187L;
    static final String DefaultLocalPropertyKey = "LastAccessedFolder";

    ArrayList<java.awt.event.ActionListener> m_actionlisteners = new ArrayList<ActionListener>();
    private String[] m_extensions = new String[]{"*"};
    private String m_description = "Files";
    File    m_file;
    private String m_LocalPropertyKey = DefaultLocalPropertyKey;

    private boolean m_autoAddExtension = true;
    private boolean m_doLoad = true;
    protected boolean m_directoryOnly = false;

    /** Creates new form FileBrowser */
    public FileBrowser() {
        initComponents();
    }

    @Override
    public void setToolTipText(String text) {
        super.setToolTipText(text); //To change body of generated methods, choose Tools | Templates.
        txtFilePath.setToolTipText(text);
        btnSelect.setToolTipText(text);
    }
    
    @Override
    public void setEnabled(boolean e) {
        super.setEnabled(e);
        btnSelect.setEnabled(e);
        txtFilePath.setEnabled(e);
    }

    protected void doActionPerformed() {
        ActionEvent ae = new ActionEvent(this, ActionEvent.ACTION_PERFORMED, "file selected",Calendar.getInstance().getTimeInMillis(), 0);
        for (java.awt.event.ActionListener al : m_actionlisteners) {
            al.actionPerformed(null);
        }
    }

    public void addActionListener(java.awt.event.ActionListener al) {
        this.m_actionlisteners.add(al);
    }

    public void setFile(String path) {
        if (path.length() == 0) {
            unsetFile();
        } else {
            setFile(new File(path));
        }
    }

    public void setFile(File path) {
        m_file = path;
        this.txtFilePath.setText(path.getAbsolutePath());
        if (path  == null) {
            return;
        }
        if ((path.exists()) &&  path.isDirectory()) {
            LocalProperties.setFolder(m_LocalPropertyKey, path);
        } else {
            LocalProperties.setFolder(m_LocalPropertyKey, path.getParent());
        }

        doActionPerformed();
    }

    public void unsetFile() {
        m_file = null;
        doActionPerformed();
    }

    public void detectFrames() {
        if (m_LocalPropertyKey == DefaultLocalPropertyKey) {
            Component p = this;
            String pathKey;
            while (p.getParent() != null) {
                p = p.getParent();
            }
            if (p instanceof JFrame) {
                pathKey = ((JFrame) p).getTitle();
                m_LocalPropertyKey = pathKey;
            }
        }
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        btnSelect = new javax.swing.JButton();
        txtFilePath = new javax.swing.JTextField();

        addAncestorListener(new javax.swing.event.AncestorListener() {
            public void ancestorMoved(javax.swing.event.AncestorEvent evt) {
            }
            public void ancestorAdded(javax.swing.event.AncestorEvent evt) {
                formAncestorAdded(evt);
            }
            public void ancestorRemoved(javax.swing.event.AncestorEvent evt) {
            }
        });
        setLayout(new java.awt.BorderLayout());

        btnSelect.setText("...");
        btnSelect.setMargin(new java.awt.Insets(2, 5, 2, 5));
        btnSelect.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnSelectActionPerformed(evt);
            }
        });
        add(btnSelect, java.awt.BorderLayout.LINE_END);

        txtFilePath.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                txtFilePathActionPerformed(evt);
            }
        });
        txtFilePath.addFocusListener(new java.awt.event.FocusAdapter() {
            public void focusLost(java.awt.event.FocusEvent evt) {
                txtFilePathFocusLost(evt);
            }
        });
        add(txtFilePath, java.awt.BorderLayout.CENTER);
    }// </editor-fold>//GEN-END:initComponents

    private void btnSelectActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnSelectActionPerformed
        String file = null;
        if (m_doLoad) {
            if (m_directoryOnly) {
                file = GetFile.getFolder(LocalProperties.getFolder(m_LocalPropertyKey).getAbsolutePath());
            } else {
                file = GetFile.getFile( getExtensions(), getDescription(), LocalProperties.getFolder(m_LocalPropertyKey).getAbsolutePath(),this);
            }
        }else {
            if (m_directoryOnly) {
                file = GetFile.getFolder(LocalProperties.getFolder(m_LocalPropertyKey).getAbsolutePath(),this);
            } else {
                file = GetFile.saveFile( getExtensions(), getDescription(), LocalProperties.getFolder(m_LocalPropertyKey).getAbsolutePath(),this);
            }
        }
        
        if (file != null) {
            if (m_autoAddExtension && !file.contains(".") && getExtensions().length >0 && !new File(file).exists()) {
                for (String ext : getExtensions()) {
                    if (!ext.contentEquals("*")) {
                        if (ext.startsWith("*")) {
                            ext = ext.substring(1);
                        }
                        if (ext.startsWith(".")) {
                            ext = ext.substring(1);
                        }
                        file = file + "." + ext;
                        break;
                    }
                }
                
            }
            txtFilePath.setText(file);
            setFile(file);
        }

    }//GEN-LAST:event_btnSelectActionPerformed

    private void txtFilePathActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_txtFilePathActionPerformed
        setFile(txtFilePath.getText());
    }//GEN-LAST:event_txtFilePathActionPerformed

    private void formAncestorAdded(javax.swing.event.AncestorEvent evt) {//GEN-FIRST:event_formAncestorAdded
        detectFrames();
    }//GEN-LAST:event_formAncestorAdded

    private void txtFilePathFocusLost(java.awt.event.FocusEvent evt) {//GEN-FIRST:event_txtFilePathFocusLost
        if (txtFilePath.getText().length() == 0) { 
            unsetFile();
        } else {
            setFile(txtFilePath.getText());
        }
    }//GEN-LAST:event_txtFilePathFocusLost


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton btnSelect;
    private javax.swing.JTextField txtFilePath;
    // End of variables declaration//GEN-END:variables

    /**
     * @return the m_LocalPropertyKey
     */
    public String getLocalPropertyKey() {
        return m_LocalPropertyKey;
    }

    /**
     * @param LocalPropertyKey the LocalPropertyKey to set
     */
    public void setLocalPropertyKey(String LocalPropertyKey) {
        this.m_LocalPropertyKey = LocalPropertyKey;
    }

    /**
     * @return the extensions
     */
    public String[] getExtensions() {
        return m_extensions;
    }

    /**
     * @param extensions the extensions to set
     */
    public void setExtensions(String[] extensions) {
        this.m_extensions = extensions;
    }

    /**
     * @return the description
     */
    public String getDescription() {
        return m_description;
    }

    /**
     * @param description the description to set
     */
    public void setDescription(String description) {
        this.m_description = description;
    }


    public String getText() {
        return txtFilePath.getText();
    }

    public void setText(String filePath) {
        txtFilePath.setText(filePath);
        m_file = new File(filePath);
    }

    public String getButtonText() {
        return btnSelect.getText();
    }

    public void setButtonText(String filePath) {
        btnSelect.setText(filePath);
    }

    public void setLoad() {
        m_doLoad = true;
    }

    public void setLoad(boolean showLoad) {
        m_doLoad = showLoad;
    }
    
    public boolean getLoad() {
        return m_doLoad;
    }
    
    public void setSave() {
        m_doLoad = false;
    }

    public File getFile() {
        return m_file;
    }

    /**
     * @return the m_directoryOnly
     */
    public boolean getDirectoryOnly() {
        return m_directoryOnly;
    }

    /**
     * @param m_directoryOnly the m_directoryOnly to set
     */
    public void setDirectoryOnly(boolean m_directoryOnly) {
        this.m_directoryOnly = m_directoryOnly;
    }

    /**
     * @return the m_autoAddExtension
     */
    public boolean isAutoAddExtension() {
        return m_autoAddExtension;
    }

    /**
     * @param m_autoAddExtension the m_autoAddExtension to set
     */
    public void setAutoAddExtension(boolean m_autoAddExtension) {
        this.m_autoAddExtension = m_autoAddExtension;
    }


}
