/* 
 * Copyright 2016 Lutz Fischer <l.fischer@ed.ac.uk>.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package rappsilber.gui.components.filter;

import java.awt.event.KeyEvent;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.text.ParseException;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.JOptionPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.table.DefaultTableModel;
import rappsilber.config.AbstractRunConfig;
import rappsilber.gui.components.AutoAddTableModelListener;
import rappsilber.ms.ToleranceUnit;
import rappsilber.ms.dataAccess.filter.spectrafilter.ScanFilteredSpectrumAccess;
import rappsilber.ms.dataAccess.msm.AbstractMSMAccess;
import rappsilber.ms.spectra.Spectra;

/**
 *
 * @author Lutz Fischer <l.fischer@ed.ac.uk>
 */
public class ScanFilterComponent extends javax.swing.JPanel {

    /**
     * Creates new form ScanFilterComponent
     */
    public ScanFilterComponent() {
        initComponents();
        fbScanFilterFile.setDescription("CSV-file/PeakList");
        fbScanFilterFile.setExtensions(new String[]{".csv","*.msm","*.mgf","*.apl","*.msmlist","*.zip"});
        // replace the deault (nicly gui-generated) table model with a new one
        tblScanFilter.setColumnSelectionAllowed(false);
        tblScanFilter.setRowSelectionAllowed(false);
        tblScanFilter.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        tblScanFilter.getModel().addTableModelListener(new AutoAddTableModelListener());        
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        buttonGroup1 = new javax.swing.ButtonGroup();
        jPanel2 = new javax.swing.JPanel();
        jScrollPane2 = new javax.swing.JScrollPane();
        tblScanFilter = new javax.swing.JTable();
        fbScanFilterFile = new rappsilber.gui.components.FileBrowser();
        btnReadScanFilter = new javax.swing.JButton();
        btnSpectraFilterClear = new javax.swing.JButton();
        rdWhiteList = new javax.swing.JRadioButton();
        rdBlackList = new javax.swing.JRadioButton();

        tblScanFilter.setModel(new javax.swing.table.DefaultTableModel(
            new Object [][] {
                {null, null}
            },
            new String [] {
                "Run", "Scan"
            }
        ) {
            Class[] types = new Class [] {
                java.lang.String.class, java.lang.Integer.class
            };

            public Class getColumnClass(int columnIndex) {
                return types [columnIndex];
            }
        });
        tblScanFilter.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyReleased(java.awt.event.KeyEvent evt) {
                tblScanFiltertblFilterKeyReleased(evt);
            }
        });
        jScrollPane2.setViewportView(tblScanFilter);

        btnReadScanFilter.setText("Read");
        btnReadScanFilter.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnReadScanFilterActionPerformed(evt);
            }
        });

        btnSpectraFilterClear.setText("Clear");
        btnSpectraFilterClear.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnSpectraFilterClearActionPerformed(evt);
            }
        });

        buttonGroup1.add(rdWhiteList);
        rdWhiteList.setSelected(true);
        rdWhiteList.setText("Include Selected");

        buttonGroup1.add(rdBlackList);
        rdBlackList.setText("Exclude Selected");

        javax.swing.GroupLayout jPanel2Layout = new javax.swing.GroupLayout(jPanel2);
        jPanel2.setLayout(jPanel2Layout);
        jPanel2Layout.setHorizontalGroup(
            jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel2Layout.createSequentialGroup()
                .addGroup(jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(jPanel2Layout.createSequentialGroup()
                        .addContainerGap()
                        .addGroup(jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                            .addComponent(jScrollPane2, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, 565, Short.MAX_VALUE)
                            .addGroup(jPanel2Layout.createSequentialGroup()
                                .addComponent(btnSpectraFilterClear)
                                .addGap(18, 18, 18)
                                .addComponent(fbScanFilterFile, javax.swing.GroupLayout.DEFAULT_SIZE, 400, Short.MAX_VALUE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(btnReadScanFilter))))
                    .addGroup(jPanel2Layout.createSequentialGroup()
                        .addGap(25, 25, 25)
                        .addComponent(rdWhiteList)
                        .addGap(18, 18, 18)
                        .addComponent(rdBlackList)))
                .addContainerGap())
        );
        jPanel2Layout.setVerticalGroup(
            jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, jPanel2Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(rdWhiteList)
                    .addComponent(rdBlackList))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addComponent(jScrollPane2, javax.swing.GroupLayout.DEFAULT_SIZE, 212, Short.MAX_VALUE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addGroup(jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(btnReadScanFilter)
                    .addComponent(btnSpectraFilterClear)
                    .addComponent(fbScanFilterFile, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap())
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 589, Short.MAX_VALUE)
            .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(layout.createSequentialGroup()
                    .addGap(0, 0, 0)
                    .addComponent(jPanel2, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addGap(0, 0, 0)))
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 300, Short.MAX_VALUE)
            .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(layout.createSequentialGroup()
                    .addGap(0, 0, 0)
                    .addComponent(jPanel2, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addGap(0, 0, 0)))
        );
    }// </editor-fold>//GEN-END:initComponents

    private void btnSpectraFilterClearActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnSpectraFilterClearActionPerformed
        DefaultTableModel tm = (DefaultTableModel) tblScanFilter.getModel();
        for (int r = tm.getRowCount() - 2; r>=0;r--) {
            tm.removeRow(r);
        }
    }//GEN-LAST:event_btnSpectraFilterClearActionPerformed

    public ScanFilteredSpectrumAccess getScanFilter() {
        ScanFilteredSpectrumAccess fsa = new ScanFilteredSpectrumAccess(rdWhiteList.isSelected());
        DefaultTableModel tm = (DefaultTableModel) tblScanFilter.getModel();
        int count = 0;
        for (int i = 0; i< tm.getRowCount() - 1; i++) {
            if (tm.getValueAt(i, 0) != null&& tm.getValueAt(i, 1) != null) {
                Integer scan = null;
                String sscan  =tm.getValueAt(i, 1).toString().trim();
                if (sscan.contentEquals("*")) { 
                    scan = null;
                } else {
                    scan = Integer.valueOf(sscan);
                }
                fsa.SelectScan(tm.getValueAt(i, 0).toString(), scan);
                count ++;
            }
        }
        if (count>0) {
            return fsa;
        } else {
            return null;
        }
    }    
    
    private void btnReadScanFilterActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnReadScanFilterActionPerformed
        DefaultTableModel tm = ((DefaultTableModel)tblScanFilter.getModel());

        String file  = fbScanFilterFile.getText();
        
        if (file.toLowerCase().matches(".*\\.(apl|zip|msm|mgf|msmlist)$")) {
            try {
                // seems we have a peaklist
                int editRow = tm.getRowCount() - 1;
                AbstractMSMAccess msm = null;
                try {
                    msm = AbstractMSMAccess.getMSMIterator(file, new ToleranceUnit("0ppm"), 0,AbstractRunConfig.DUMMYCONFIG);
                } catch (ParseException ex) {
                    Logger.getLogger(ScanFilterComponent.class.getName()).log(Level.SEVERE, null, ex);
                    JOptionPane.showMessageDialog(this, ex.getLocalizedMessage(), "error parsing input file", JOptionPane.ERROR_MESSAGE);
                    return;
                }
                while (msm.hasNext()) {
                    Spectra s = msm.next();
                    tm.setValueAt(s.getRun(), editRow, 0);
                    tm.setValueAt(s.getScanNumber(), editRow++, 1);                    
                }
                
            } catch (FileNotFoundException ex) {
                Logger.getLogger(ScanFilterComponent.class.getName()).log(Level.SEVERE, null, ex);
            } catch (IOException ex) {
                Logger.getLogger(ScanFilterComponent.class.getName()).log(Level.SEVERE, null, ex);
            }
            
        } else if (file != null || file.length() >0) {
            File in = new File(file);
            if (!in.exists()) {
                JOptionPane.showMessageDialog(this, "file " + file + " not found", "File Not Found", JOptionPane.ERROR_MESSAGE);
                return;
            }
            BufferedReader br;
            try {
                br = new BufferedReader(new FileReader(in));
            } catch (FileNotFoundException ex) {
                JOptionPane.showMessageDialog(this, "file " + file + " not found", "File Not Found", JOptionPane.ERROR_MESSAGE);
                return;
            }
            String line;
            try {
                int editRow = tm.getRowCount() - 1;
                Pattern p = Pattern.compile("^\\s*(?:\\\")?([^\",]*)(?:\\\")?\\s*,\\s*(?:\\\")?([0-9]+\\*)(?:.0)?(?:\\\")?\\s*(?:,.*)?$");
                while ((line = br.readLine()) != null) {
                    Matcher m = p.matcher(line);
                    if (m.matches()) {
                        String[] data = line.split(",",3);
                        tm.setValueAt(m.group(1), editRow, 0);
                        tm.setValueAt(m.group(2), editRow++, 1);
                    }
                }
            } catch (IOException ex) {
                JOptionPane.showMessageDialog(this, "Error while reading file " + file + " !", "File Not Found", JOptionPane.ERROR_MESSAGE);
                return;
            }
        }
    }//GEN-LAST:event_btnReadScanFilterActionPerformed

    private void tblScanFiltertblFilterKeyReleased(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_tblScanFiltertblFilterKeyReleased
        if(evt.getKeyCode() == KeyEvent.VK_DELETE) {
            JTable tbl = (JTable) evt.getSource();
            if (tbl.getSelectedRowCount() > 0) {
                DefaultTableModel tm = (DefaultTableModel) tbl.getModel();
                int LastRow = tm.getRowCount();
                int[] rows = tbl.getSelectedRows();
                for (int r = rows.length;r-->0;) {
                    if (r<LastRow) {
                        tm.removeRow(r);
                    }
                }

            }
        }
    }//GEN-LAST:event_tblScanFiltertblFilterKeyReleased

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton btnReadScanFilter;
    private javax.swing.JButton btnSpectraFilterClear;
    private javax.swing.ButtonGroup buttonGroup1;
    private rappsilber.gui.components.FileBrowser fbScanFilterFile;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JScrollPane jScrollPane2;
    private javax.swing.JRadioButton rdBlackList;
    private javax.swing.JRadioButton rdWhiteList;
    private javax.swing.JTable tblScanFilter;
    // End of variables declaration//GEN-END:variables
}
